/*
 * Copyright (c) 2006-2021, RT-Thread Development Team
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author       Notes
 */
#ifndef __AUTH_H__
#define __AUTH_H__

#include <rpc/xdr.h>

/*
 * Status returned from authentication check
 */
enum auth_stat {
    AUTH_OK=0,
    /*
     * failed at remote end
     */
    AUTH_BADCRED=1,         /* bogus credentials (seal broken) */
    AUTH_REJECTEDCRED=2,        /* client should begin new session */
    AUTH_BADVERF=3,         /* bogus verifier (seal broken) */
    AUTH_REJECTEDVERF=4,        /* verifier expired or was replayed */
    AUTH_TOOWEAK=5,         /* rejected due to security reasons */
    /*
     * failed locally
    */
    AUTH_INVALIDRESP=6,     /* bogus response verifier */
    AUTH_FAILED=7           /* some unknown reason */
};

union des_block {
    struct {
        uint32_t high;
        uint32_t low;
    } key;
    char c[8];
};
typedef union des_block des_block;

/*
 * Authentication info.  Opaque to client.
 */
struct opaque_auth {
    enum_t  oa_flavor;      /* flavor of auth */
    char*   oa_base;        /* address of more auth stuff */
    unsigned int    oa_length;      /* not to exceed MAX_AUTH_BYTES */
};

/*
 * Auth handle, interface to client side authenticators.
 */
typedef struct AUTH AUTH;
struct AUTH {
  struct opaque_auth ah_cred;
  struct opaque_auth ah_verf;
  union des_block ah_key;
  struct auth_ops {
    void (*ah_nextverf) (AUTH *);
    int  (*ah_marshal) (AUTH *, XDR *);     /* nextverf & serialize */
    int  (*ah_validate) (AUTH *, struct opaque_auth *);
                        /* validate verifier */
    int  (*ah_refresh) (AUTH *);        /* refresh credentials */
    void (*ah_destroy) (AUTH *);            /* destroy this structure */
  } *ah_ops;
  char* ah_private;
};

extern struct opaque_auth _null_auth;


/*
 * Authentication ops.
 * The ops and the auth handle provide the interface to the authenticators.
 *
 * AUTH *auth;
 * XDR  *xdrs;
 * struct opaque_auth verf;
 */
#define AUTH_NEXTVERF(auth)     \
        ((*((auth)->ah_ops->ah_nextverf))(auth))
#define auth_nextverf(auth)     \
        ((*((auth)->ah_ops->ah_nextverf))(auth))

#define AUTH_MARSHALL(auth, xdrs)   \
        ((*((auth)->ah_ops->ah_marshal))(auth, xdrs))
#define auth_marshall(auth, xdrs)   \
        ((*((auth)->ah_ops->ah_marshal))(auth, xdrs))

#define AUTH_VALIDATE(auth, verfp)  \
        ((*((auth)->ah_ops->ah_validate))((auth), verfp))
#define auth_validate(auth, verfp)  \
        ((*((auth)->ah_ops->ah_validate))((auth), verfp))

#define AUTH_REFRESH(auth)      \
        ((*((auth)->ah_ops->ah_refresh))(auth))
#define auth_refresh(auth)      \
        ((*((auth)->ah_ops->ah_refresh))(auth))

#define AUTH_DESTROY(auth)      \
        ((*((auth)->ah_ops->ah_destroy))(auth))
#define auth_destroy(auth)      \
        ((*((auth)->ah_ops->ah_destroy))(auth))

#define MAX_AUTH_BYTES  400
#define MAXNETNAMELEN   255 /* maximum length of network user's name */

AUTH *authnone_create(void);

#endif
